function [indices, totalVar, moreOutput] = NJW(X,K,OPTIONS)
%
% Spectral Clustering by Ng, Jordan and Weiss (NIPS 01')
%
% USAGE
%   indices = NJW(X,K,OPTIONS)
%
% INPUT
%   X: NxD data matrix, normalized to be within the cube [0,1]^D
%   K: number of clusters
%   OPTIONS: a structure array of the following optional parameters:
%     .type: the type of the parameter sigma used in NJW
%       'value': a single given value (default)
%       'knn': use nearest neighbors to infer sigma from data
%       'local_scaling': self-tuning spectral clustering 
%                        by Zelnik-Manor & Perona
%     .sigma: real number representing scale (when .type ='value'),
%             or integer = number of nearest numbers (otherwise)
%
% OUTPUT
%   indices: a vector of group labels of the data points
%   totalVar: smallest variance in the eigenspace corresponding to
%             different parameter values of sigma
%
%%
N = size(X,1);

% if ~isfield(OPTIONS, 'subset')
%     OPTIONS.subset = 1:N;
% end

%%
G = X*X';
norms = diag(G);
dists = repmat(norms,1,N) + repmat(norms',N,1) - (G+G);

% norms = sum(X.^2,2);
% Kern  = X*X(OPTIONS.subset,:)';
% dists = (repmat(norms,1,numel(OPTIONS.subset)) + repmat(norms(OPTIONS.subset)',N,1)) - (Kern+Kern);

%%
if ~isfield(OPTIONS,'type')
    OPTIONS.type = 'value';
end

switch OPTIONS.type
    case 'value'
        isigma = num2cell((0.5)./((OPTIONS.sigma).^2));
    case 'knn'
        dists_sort = sort(reshape(dists,[],1),'ascend');
        isigma = num2cell(0.5./dists_sort((OPTIONS.sigma+1)*N,1));
    case 'local_scaling'
        sorted_dists = sort(dists,2,'ascend');
        k_sigmas = length(OPTIONS.sigma);
        isigma = cell(1, k_sigmas);
        for k = 1:k_sigmas
            temp = 1./sqrt(sorted_dists(:,OPTIONS.sigma(k)+1));
            isigma{k} = repmat(temp,1,N).*repmat(temp',N,1);
        end
end

%%
if nargout>2
    moreOutput = struct();
    moreOutput.dists = dists;
    moreOutput.optSigma = [];
    moreOutput.optW = [];
    moreOutput.optZ = [];
    moreOutput.optU = [];
    moreOutput.optEigvals = [];
end

%%
totalVar = Inf; 
indices = [];
for k = 1:length(isigma)
    
    W = exp(-dists.*isigma{k});
    W(1:(N+1):N^2) = 0;
      
    degrees = sum(W,2);
    degrees(degrees==0) = 1;
    D_inv_sqrt = repmat(1./sqrt(degrees),1,N);
    Z = D_inv_sqrt.*W.*D_inv_sqrt';
    
    Z = real(Z); Z = (Z+Z')/2;
    
    [Ui,eigvals] = eigs(Z,2*K,'LM',struct('isreal', true, 'issym', true, 'disp', 0));
    eigvals = diag(eigvals);
    
    %[~, ind_sort] = sort(abs(eigvals),'descend');
    %U = Ui(:,ind_sort(1:K));
    U = Ui(:,1:K);
    U = U./repmat(sqrt(sum(U.^2,2)),1,K);
    
    % apply Kmeans 10 times and use the best result
    indices1 = zeros(N,1);
    totalVar1 = Inf;
    for j = 1:10
        [inds,~,SUMD] = kmeans(U,K,'EmptyAction','drop');
        if totalVar1 > sum(SUMD)
            indices1 = inds;
            totalVar1 = sum(SUMD);
        end
    end
    
    if totalVar1 < totalVar
        totalVar = totalVar1;
        indices = indices1;
        if nargout>2
            moreOutput.optSigma = OPTIONS.sigma(k);
            moreOutput.optW = W;
            moreOutput.optZ = Z;
            moreOutput.optU = U;
            moreOutput.optEigvals = eigvals;
        end
    end
    
end

