function [indices, degGap, moreOutput] = NJW_outliers(X,alpha,OPTIONS)

%
% Spectral Clustering by Ng, Jordan, and Weiss
% when outliers are present in the data
%
% USAGE 
%   indices = NJW_outliers(X,alpha,OPTIONS)
%
% INPUT
%   X: NxD data matrix, normalized within the unit cube [0,1]^D
%   alpha: number of outliers (if >=1); percentage of outliers (if <1)
%   OPTIONS: a structure array of the following optional parameters:
%     .type: the type of the parameter sigma used in NJW
%       'value': a single given value (default)
%       'knn': use nearest neighbors to infer sigma from data
%       'local_scaling': self-tuning spectral clustering 
%                        by Zelnik-Manor & Perona
%     .sigma: real number representing scale (when .type ='value');
%             integer representing number of nearest numbers (otherwise)
%
% OUTPUT
%   indices: a vector of group labels of the data points 
%   degGap: largest degree gap between inliers and outliers

%%
N = size(X,1);

if alpha<1, %percentage of outliers
    alpha = round(N*alpha);
end

if ~isfield(OPTIONS, 'subset')
    OPTIONS.subset = 1:N;
end

%%
% Kern = X*X';
% norms = diag(Kern);
% dists = (repmat(norms,1,N) + repmat(norms',N,1)) - (Kern+Kern);

norms = sum(X.^2,2);
Kern = X*X(OPTIONS.subset,:)';
dists = (repmat(norms,1,numel(OPTIONS.subset)) + repmat(norms(OPTIONS.subset)',N,1)) - (Kern+Kern);

%%
if ~isfield(OPTIONS,'type')
    OPTIONS.type = 'value';
end

switch OPTIONS.type
    case 'value'
        isigma = num2cell(0.5./((OPTIONS.sigma).^2));
    case 'knn'
        dists_sort = sort(reshape(dists,[],1),'ascend');
        isigma = num2cell(0.5./dists_sort((OPTIONS.sigma+1)*N,1));
    case 'local_scaling'
        sorted_dists = sort(dists,2,'ascend');
        k_sigmas = length(OPTIONS.sigma);
        isigma = cell(1, k_sigmas);
        for k = 1:k_sigmas
            temp = 1./sqrt(sorted_dists(:,OPTIONS.sigma(k)+1));
            isigma{k} = repmat(temp,1,N).*repmat(temp',N,1);
        end
end

%%
if nargout>2
    moreOutput = struct();
    moreOutput.dists = dists;
    moreOutput.optSigma = [];
    moreOutput.optW = [];
end

%% remove outliers   
degGap = 0; 
indices = [];

for k = 1:length(isigma)
    
    W = exp(-dists.*isigma{k});
    
    degrees = sum(W,2);
    [~, ind_sort] = sort(degrees,'ascend');
    
    inliers = ind_sort(alpha+1:end);
    
    indices1 = zeros(N,1);
    indices1(inliers) = 1;
    
    degGap1 = (mean(degrees(inliers)) - mean(degrees(ind_sort(1:alpha))))/degrees(ind_sort(end));
    
    if  degGap < degGap1
        degGap = degGap1;
        indices = indices1;
        if nargout>2
            moreOutput.optSigma = OPTIONS.sigma(k);
            moreOutput.optW = W;
        end
    end
    
end

